<?php

declare(strict_types=1);

namespace Erlage\Photogram\Pattern;

use Exception;
use Throwable;
use RedBeanPHP\OODBBean;
use Erlage\Photogram\System;
use Erlage\Photogram\Request;
use Erlage\Photogram\Session;
use Erlage\Photogram\Response;
use Erlage\Photogram\Settings;
use Erlage\Photogram\AdminSession;
use Erlage\Photogram\SystemLogger;
use Erlage\Photogram\Structures\TableDataDock;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Data\Common\CommonQueries;
use Erlage\Photogram\Data\Models\AbstractModel;
use Erlage\Photogram\Exceptions\BreakException;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Exceptions\ServerException;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Exceptions\SessionException;
use Erlage\Photogram\Data\Models\Admin\AdminModel;
use Erlage\Photogram\Exceptions\ModelDataException;
use Erlage\Photogram\Structures\TableDependencyGraph;

abstract class ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | user auth related
    |--------------------------------------------------------------------------
    */

    /**
     * @var Session
     */
    public static $userSession;

    /**
     * @var UserModel
     */
    public static $authedUserModel;

    /*
    |--------------------------------------------------------------------------
    | admin auth related
    |--------------------------------------------------------------------------
    */

    /**
     * @var AdminSession
     */
    public static $adminSession;

    /**
     * @var AdminModel
     */
    public static $authedAdminModel;

    /**
     * Whether to process current request in admin mode.
     * 
     * @var bool
     */
    public static $flagAdminMode = false;

    /*
    |--------------------------------------------------------------------------
    | response data
    |--------------------------------------------------------------------------
    */

    /**
     * @var Request
     */
    public static $request;

    /**
     * @var Response
     */
    public static $response;

    /**
     * @var TableDataDock
     */
    protected static $dataDock;

    /**
     * @var TableDataDock
     */
    protected static $additionalDataDock;

    /**
     * @var TableDependencyGraph
     */
    protected static $dependencyGraph;

    /**
     * @var string
     */
    protected static $message = ResponseConstants::SUCCESS_MSG;

    /**
     * Static init. It's like a constructor but for static properties.
     * It should be called by the wrapping method process()
     * @see self::process()
     */
    public static function initDocks(): void
    {
        self::$dataDock = new TableDataDock();

        self::$additionalDataDock = new TableDataDock();

        self::$dependencyGraph = new TableDependencyGraph();
    }

    /**
     * Main wrapper. it wraps the entire logic of processing client request
     * inside a environment full of utilities. Exception thrown by the processing
     * logic will be handled gracefully and client will notified along with
     * valuable information to help them take correct measure
     * @throws Exception 
     */
    final public static function process(callable $procedure): void
    {
        try
        {
            self::initDocks();

            self::$request = Request::instance();
            self::$response = Response::instance();
            self::$userSession = Session::instance();
            self::$adminSession = AdminSession::instance();

            if (self::$flagAdminMode)
            {
                self::adminAuthenticate();
            }
            else
            {
                self::userAuthenticate();
            }

            // request controller

            try
            {
                $procedure();

                self::processDependencies();

                self::$response -> setMessage(self::$message);

                if (self::$flagAdminMode)
                {
                    self::$adminSession -> updateSession();
                }
                else
                {
                    self::$userSession -> updateSession();
                }
            }
            catch (SessionException $sessionProblem)
            {
                // clear stale data
                self::initDocks();

                // in case of a session exception make sure to send user object
                if (self::$userSession -> isAuthenticated())
                {
                    self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());
                }

                self::$response -> setMessage($sessionProblem -> getMessage());
            }

            // dispatch data containers

            while (self::$dataDock -> canDispatch() && $container = self::$dataDock -> dispatchContainer())
            {
                self::$response -> setContent($container -> getTableName(), $container -> getDataMaps());
            }

            // dispatch additional containers

            while (self::$additionalDataDock -> canDispatch() && $container = self::$additionalDataDock -> dispatchContainer())
            {
                self::$response -> setAdditionalContent($container -> getTableName(), $container -> getDataMaps());
            }
        }
        catch (ModelDataException $exception)
        {
            self::$response -> setMessage(ResponseConstants::ERROR_BAD_REQUEST_MSG);
        }
        catch (RequestException $exception)
        {
            self::$response -> setMessage($exception -> getMessage());
        }
        catch (ServerException $exception)
        {
            self::$response -> setMessage(ResponseConstants::ERROR_FAILED_REQUEST_MSG);
        }
        catch (Exception|Throwable $exception)
        {
            SystemLogger::internalException($exception);

            self::$response -> setMessage(ResponseConstants::ERROR_FAILED_REQUEST_MSG);
        }
        finally
        {
            self::$response -> dispatch();
        }
    }

    /**
     * Sets message that will be sent inside response.
     * if called multiple times, only message from last call will
     * be dispatched at the time of dispatch and rest will be ignored 
     */
    final public static function setMessage(string $message): void
    {
        self::$message = $message;
    }

    /**
     * @param OODBBean[] $beans 
     */
    final public static function processBeans(string $tableName, array $beans, ?callable $customProcedure = null): void
    {
        foreach ($beans as $bean)
        {
            $abstractTableClass = System::getModelClassFromTableName($tableName);

            $model = $abstractTableClass::createFromUntouchedBean_noException($bean);

            try
            {
                if ($model -> isModel())
                {
                    if (null != $customProcedure)
                    {
                        $customProcedure($model);
                    }

                    self::addToResponse($tableName, $model -> getDataMap());
                }
            }
            catch (BreakException $e)
            {
                // $customProcedure($model) can throw [BreakException] to indicate that
                // current model should not be added in the response
                // therefore we don't do anything here other than catching exception
                // and ignoring the model
            }
        }
    }

    final public static function addToResponse(string $tableName, array $map): void
    {
        $container = self::$dataDock -> getContainer($tableName);

        $container -> addDataMap($map);
    }

    final public static function addToAdditionalResponse(string $tableName, array $map): void
    {
        $container = self::$additionalDataDock -> getContainer($tableName);

        $container -> addDataMap($map);
    }

    final public static function addDependency(string $tableName, string $id): void
    {
        $dependencyNode = self::$dependencyGraph -> getDependencyNode($tableName);

        $dependencyNode -> add($id);
    }

    /*
    |--------------------------------------------------------------------------
    | fetch dependencies
    |--------------------------------------------------------------------------
    */

    final public static function processDependencies(): void
    {
        while (self::$dependencyGraph -> canPop() && $dependencyNode = self::$dependencyGraph -> popNode())
        {
            $abstractTableClass = System::getTableClassFromTableName($dependencyNode -> getTableName());

            self::fetchModelsAndAdd(
                $dependencyNode -> getTableName(),
                array($abstractTableClass::getPrimaryAttribute() => $dependencyNode -> getIds())
            );
        }

        self::$dependencyGraph -> clear();
    }

    /*
    |--------------------------------------------------------------------------
    | fetch dependencies
    |--------------------------------------------------------------------------
    */

    final public static function fetchModelsAndAdd(string $tableName, array $predicates): void
    {
        /**
         * @var AbstractModel[]
         * Note: AbstractModel is not a RuntimeType. Actual type will always be a SuperType.
         * AbtractModel contains methods that are used here so marking this help resolves those calls
         */
        $models = CommonQueries::modelsWithMatchingPredicates($tableName, $predicates);

        foreach ($models as $model)
        {
            self::addToResponse($tableName, $model -> getDataMap());
        }
    }

    /*
    |--------------------------------------------------------------------------
    |  additional dependencies helper
    |--------------------------------------------------------------------------
    */

    final public static function fetchModelsAndAddAsAdditional(string $tableName, array $predicates): void
    {
        /**
         * @var AbstractModel[]
         * Note: AbstractModel is not a RuntimeType. Actual type will always be a SuperType.
         * AbtractModel contains methods that are used here so marking this help resolves those calls
         */
        $models = CommonQueries::modelsWithMatchingPredicates($tableName, $predicates);

        foreach ($models as $model)
        {
            self::addToAdditionalResponse($tableName, $model -> getDataMap());
        }
    }

    /*
    |--------------------------------------------------------------------------
    | admin auth funs
    |--------------------------------------------------------------------------
    */

    final public static function adminAuthenticate(): void
    {
        if (self::$adminSession -> isAuthenticated())
        {
            self::$authedAdminModel = self::$adminSession -> getAdminModel();
        }
    }

    final public static function adminEnsureAuthenticated(): void
    {
        if ( ! self::$adminSession -> isAuthenticated())
        {
            throw new RequestException(ResponseConstants::D_ERROR_ADMIN_NOT_MATCHED_MSG);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | user auth funs
    |--------------------------------------------------------------------------
    */

    final public static function userAuthenticate(): void
    {
        if (self::$userSession -> isAuthenticated())
        {
            self::$authedUserModel = self::$userSession -> getUserModel();
        }
    }

    /**
     * Ensure user is authenticated. This method uses the isAuthetnicated call
     * to check whether user is logged in else it throws a exception
     * @see self::isAuthenticated() 
     * @throws SessionException 
     */
    final public static function userEnsureAuthenticated(
        string $ifNotMessage = ResponseConstants::D_ERROR_SESSION_UNAUTHORIZED_MSG
    ): void {
        /*
        |--------------------------------------------------------------------------
        | make sure user is autheticated
        |--------------------------------------------------------------------------
        */

        if ( ! self::$userSession -> isAuthenticated())
        {
            throw new SessionException($ifNotMessage);
        }

        /*
        |--------------------------------------------------------------------------
        | if admin mode, no more checks
        |--------------------------------------------------------------------------
        */

        if (self::$flagAdminMode)
        {
            return;
        }

        /*
        |--------------------------------------------------------------------------
        | if enabled email verification, make sure email is verified
        |--------------------------------------------------------------------------
        */

        if ( ! self::$authedUserModel -> isEmailVerified() && Settings::getBool(ServerConstants::SS_BOOL_USER_EMAIL_VERIFICATION))
        {
            throw new SessionException(ResponseConstants::D_ERROR_SESSION_REQUIRES_EMAIL_VERIFICATION);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | helpers
    |--------------------------------------------------------------------------
    */

    /**
     * Whether a/or all items are available.
     * @see self::ensureAvailability() for more
     * @param string[] $items
     */
    final public static function isAvailable(...$items): bool
    {
        foreach ($items as $item)
        {
            if (SystemConstants::NA === $item)
            {
                return false;
            }
        }

        return true;
    }

    /**
     * Ensure that provided model exists and is not corrupted. This is done by 'isModel'
     * and/or 'isNotModel' calls from [AbstractModel] class. In case model is corrupted
     * a [RequestException] will be thrown containing the $message. A BAD REQUEST message
     * will be used as default for $message
     * 
     */
    final public static function ensureModel(AbstractModel $model, $message = ResponseConstants::ERROR_BAD_REQUEST_MSG): void
    {
        if ($model -> isNotModel())
        {
            throw new RequestException($message);
        }
    }

    /**
     * Ensure that all provided $items are set using build-in 'isset' call. In case one
     * of the item isn't set a [RequestException] will be thrown containing the $message
     */
    final public static function ensureIsset(string $message, ...$items): void
    {
        foreach ($items as $item)
        {
            if ( ! isset($item))
            {
                throw new RequestException($message);
            }
        }
    }

    /**
     * Ensure that all $items are values. Ensuring a value means ensuring its
     * availability and actual literal value.
     * Please refer to 'ensureAvailability' & 'ensureEmptiness' calls for more
     */
    final public static function ensureValue(string $message, ...$items): void
    {
        self::ensureAvailability($message, ...$items);

        self::ensureEmptiness($message, ...$items);
    }

    /**
     * Ensure that all $items are available. Availability refers whether
     * a item is assigned to value of NA system constant. Request module
     * returns NA value for keys which aren't present in client's request
     * but are requested by the system somewhere.
     * @param string[] $items
     */
    final public static function ensureAvailability(string $message, ...$items): void
    {
        foreach ($items as $item)
        {
            if (SystemConstants::NA === $item)
            {
                throw new RequestException($message);
            }
        }
    }

    /**
     * Checks whether all $items are non-empty using isEmpty built-in call
     * If anyone of the item is empty, it throws a [RequestException].
     * $message will be passed to the throwed exception
     */
    final public static function ensureEmptiness(string $message, ...$items): void
    {
        foreach ($items as $item)
        {
            if (empty($item))
            {
                throw new RequestException($message);
            }
        }
    }
}
